#include <iostream>
#include <stdexcept>

// ---- Meta Assertions
//
// Like assert() for runtime, this can be used to
// document preconditions.  If the predicate is not
// fulfilled, we get a compile error.
//
// The trick is, C++ disallows arrays of
// constant length of zero.
//

#define STATIC_ASSERT(x) static const int STATIC_ASSERT_ ## __COUNT__ [(x)]

// ---- Runtime Error
//
// This will be thrown at runtime before some too small
// or too big value is set.  No more silent overflows.
//
struct out_of_range : virtual std::out_of_range {
  out_of_range() : std::out_of_range("value out of range") {}
};

// ---- Our Actual 'restricted' type

template<class T, T Min, T Max>
struct restricted {
  STATIC_ASSERT((Min < Max));
  
  T value;
  
  explicit restricted(T const& o)
    : value(o) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
};

// ---- How to use it

int main() {
  restricted<int, 0, 10> i(5);  
  
  //
  // copy construction still here:
  //
  restricted<int, 0, 10> j(i);

  // TODO: uncomment to test
  //
  //restricted<int, 0, 20> k(j);
  //
  // Note the different 'Max' parameter!
}
